<?php

namespace App\Http\Controllers\Api;

use App\Enums\DeliveryStatus;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\Delivery\CancelRequest;
use App\Http\Requests\Api\Delivery\CompletedPaymentRequest;
use App\Http\Requests\Api\Delivery\DeliveryListRequest;
use App\Models\CancelReason;
use App\Models\Delivery;
use App\Models\DeliveryPayment;
use App\Models\DeliveryStatusHistory;
use App\Traits\ApiResponse;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

class DeliveryController extends Controller
{
    use ApiResponse;

    /**
     * Delviery Listing
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function deliveryListing(DeliveryListRequest $request)
    {
        $validated = $request->validated();
        $branchId = $request->user()->default_branch_id;

        $data = Delivery::where('branch_id', $branchId)
            ->select('id','code','receiver_address','receiver_phone','assigned_at','amount_usd','amount_khr','status')
            ->when(Arr::get($validated, 'driver_id'), function ($query, $driver_id) {
                return $query->where('delivery_by', $driver_id);
            })
            ->when(Arr::get($validated, 'customer_id'), function ($query, $customer_id) {
                return $query->where('customer_id', $customer_id);
            })
            ->when(Arr::get($validated, 'status'), function ($query, $status) {
                return $query->where('status', $status);
            })
            ->when(Arr::get($validated, 'date'), function ($query, $date) {
                $dates = explode(' to ', $date);

                if (count($dates) === 2) {
                    $from = $dates[0].' 00:00:00';
                    $to = $dates[1].' 23:59:59';

                    return $query->whereBetween('assigned_at', [$from, $to]);
                } elseif (count($dates) === 1 && $dates[0]) {
                    return $query->whereBetween('assigned_at', [
                        $dates[0].' 00:00:00',
                        $dates[0].' 23:59:59',
                    ]);
                }

                return $query;
            })
            ->orderBy('assigned_at', 'desc')
            ->paginate(10);

        return $this->successResponse($data, 'delivery listing');
    }


    /**
     * Details
     */
    public function details($id)
    {
        $delivery = Delivery::with('customer','payments')
            ->where('id', $id)
            ->firstOrFail();

        $data = [
            'code' => $delivery->code,
            'assigned_at' => $delivery->assigned_at,
            'receiver_address' => $delivery->receiver_address,
            'receiver_phone' => $delivery->receiver_phone,
            'original_amount_usd' => $delivery->amount_usd,
            'original_amount_khr' => $delivery->amount_khr,
            'completed_at' => $delivery->completed_at,
            'status' => $delivery->status,
            'note' => $delivery->note,
            'customer' => [
                'name' => $delivery->customer?->customer_name,
                'ref_code' => $delivery->customer?->ref_code,
                'phone' => $delivery->customer?->phone,
                'type_of_customer' => $delivery->customer?->type_of_customer,
            ],
            'collect_payments'=>[
                'amount_usd' => $delivery->payments
                    ->where('currency', 'usd')
                    ->sum('amount'),

                'amount_khr' => $delivery->payments
                    ->where('currency', 'khr')
                    ->sum('amount'),
            ],
        ];

        return $this->successResponse($data, 'delivery details');
    }




    public function processCompletedPayment(CompletedPaymentRequest $request)
    {
        $data = $request->validated();
        $delivery = Delivery::findOrFail($data['delivery_id']);

        if ($delivery->status === DeliveryStatus::COMPLETED->value) {
            return $this->errorResponse('This delivery is already completed.');
        }

        DB::beginTransaction();

        try {
            $delivery->status = DeliveryStatus::COMPLETED->value;
            $delivery->completed_at = now();
            $delivery->save();

            $customer = $delivery->customer?->customer_name ?? 'N/A';

            DeliveryStatusHistory::create([
                'delivery_id' => $delivery->id,
                'status' => DeliveryStatus::COMPLETED->value,
                'note' => $data['note'] ?? 'Delivery completed by '.auth()->user()->display_name,
                'changed_by' => auth()->id(),
                'changed_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            foreach ($data['payments'] as $paymentTo) {
                DeliveryPayment::create([
                    'delivery_id' => $delivery->id,
                    'payment_to' => $paymentTo['payment_to'],
                    'amount' => $paymentTo['amount'],
                    'currency' => $paymentTo['currency'],
                    'payment_method' => $paymentTo['payment_method'],
                    'note' => $data['note'] ?? null,
                    'created_by' => auth()->id(),
                ]);

                sendTelegramNotification([
                    'delivery_code' => $delivery->code,
                    'customer' => $customer,
                    'payment' => $paymentTo['payment_to'],
                    'amount' => $paymentTo['amount'],
                    'currency' => $paymentTo['currency'],
                    'payment_method' => $paymentTo['payment_method'],
                ]);
            }

            DB::commit();

            return $this->successResponse($delivery, 'Delivery marked as completed with payment.');
        } catch (\Exception $e) {
            DB::rollBack();

            return $this->errorResponse('Failed to mark delivery as completed: '.$e->getMessage(), 500);
        }
    }

    public function processCancelReason(CancelRequest $request)
    {
        $data = $request->validated();

        $delivery = Delivery::findOrFail($data['delivery_id']);

        if ($delivery->status === DeliveryStatus::CANCELED->value) {
            return $this->errorResponse('This delivery is already canceled.');
        }

        DB::beginTransaction();
        try {
            $delivery->status = DeliveryStatus::CANCELED->value;
            $delivery->cancel_reason_id = $data['cancel_reason_id'];
            $delivery->cancel_note = $data['cancel_note'];
            $delivery->save();

            $reason = CancelReason::findOrFail($data['cancel_reason_id']);
            $message = $data['cancel_note'] ?? $reason->reason;

            DeliveryStatusHistory::create([
                'delivery_id' => $data['delivery_id'],
                'status' => DeliveryStatus::CANCELED->value,
                'note' => $message,
                'changed_by' => auth()->id(),
                'changed_at' => now(),
            ]);

            $customer = $delivery->customer?->customer_name ?? 'N/A';

            sendTelegramCancelNotification([
                'delivery_code' => $delivery->code,
                'customer' => $customer,
                'reason' => $message,
            ]);

            DB::commit();

            return $this->successResponse($delivery, 'Cancel reason success.');
        } catch (\Exception $e) {
            DB::rollBack();

            return $this->errorResponse('Failed to cancel delivery: '.$e->getMessage(), 500);
        }
    }
}
