<?php

namespace App\Http\Controllers\Api;

use App\Enums\DeliveryStatus;
use App\Http\Controllers\Controller;
use App\Models\Delivery;
use App\Models\DeliveryPayment;
use App\Models\DriverPayment;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Carbon\Carbon;

class DriverSummaryController extends Controller
{
    use ApiResponse;

    public function dashboard(Request $request)
    {   /**
         * For Driver Role
         */
        $driver = auth()->user();
        // Base delivery query
        $deliveryQuery = Delivery::where('delivery_by', $driver->id)
        ->where('is_active',true);

        /* ============================
         * Payments (FROM driver_payments)
         * ============================ */
            $totalOrginCollectedUSD  = (clone $deliveryQuery)
                ->where('is_paid',false)
                ->whereIn('status',[DeliveryStatus::ASSIGNED->value,DeliveryStatus::COMPLETED->value])
                ->sum('amount_usd');
         
            $totalOrginCollectedKHR  = (clone $deliveryQuery)
                ->where('is_paid',false)
                ->whereIn('status',[DeliveryStatus::ASSIGNED->value,DeliveryStatus::COMPLETED->value])
                ->sum('amount_khr');

            /* ============================
            * Currency-wise Collected Summary
            * ============================ */
            $totalCollect = DeliveryPayment::whereHas('delivery', function ($q) use ($driver) {
                $q->where('delivery_by', $driver->id)->where('is_paid',false);
            })
            ->selectRaw('currency, SUM(amount) as total_amount')
            ->where('payment_to','driver')
            ->groupBy('currency')
            ->get();


            $collectedUSD = 0;
            $collectedKHR = 0;

            foreach ($totalCollect as $row) {
                if (strtolower($row->currency) === 'usd') {
                    $collectedUSD = (float) $row->total_amount;
                }

                if (strtolower($row->currency) === 'khr') {
                    $collectedKHR = (float) $row->total_amount;
                }
            }
            $exchangeRate = getExchangeRate();

            // Original (unpaid COD)
            $originalTotalInUSD =
                (float) $totalOrginCollectedUSD +
                ((float) $totalOrginCollectedKHR / $exchangeRate);

            // Collected
            $collectedTotalInUSD =
                $collectedUSD +
                ($collectedKHR / $exchangeRate);

            // Outstanding
            $outstanding = max(
                round($originalTotalInUSD - $collectedTotalInUSD, 2),
                0
            );


        /* ============================
         * Today Summary
         * ============================ */
        $today = Carbon::today();

        $todaySummary = [
            'total_package' => (clone $deliveryQuery)
             ->where('is_paid', false)
               ->where(function ($q) use ($today) {
                    $q->whereDate('created_at', $today)
                    ->orWhereDate('assigned_at', $today)
                    ->orWhereDate('completed_at', $today)
                    ->orWhereDate('canceled_at', $today)
                    ->where('status', DeliveryStatus::ASSIGNED->value);
                })

                ->count(),

            'in_progress' =>  (clone $deliveryQuery)
                ->where('status', DeliveryStatus::ASSIGNED->value)
                ->orWhereDate('assigned_at', $today)
                ->count(),


            'completed' => (clone $deliveryQuery)
                ->where('status', 'completed')
                ->whereDate('completed_at', $today)
                ->where('is_paid', false)
                ->count(),

            'cancelled' => (clone $deliveryQuery)
                ->where('status', 'canceled')
                ->whereDate('canceled_at', $today)
                ->count(),
        ];

 

        return $this->successResponse([
            'payment' => [
                'total_original' => [
                    'usd' => (float) $totalOrginCollectedUSD,
                    'khr' => (float) $totalOrginCollectedKHR,
                ],
                'total_collected' => [
                    'usd' => $collectedUSD,
                    'khr' => $collectedKHR,
                ],
                'outstanding_usd' => $outstanding,
            ],
            'today' => $todaySummary,
        ], 'Driver dashboard summary');
    }
}
