<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\Api\Delivery\DriverPaymentSummaryRequest;
use App\Models\Delivery;
use App\Traits\ApiResponse;
use App\Models\SummaryDriverPayment;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Arr;
class DriverSummaryPaymentController extends Controller
{
    use ApiResponse;

    /**
     * Summary Invoice 
     */
    public function summaryInvoice(DriverPaymentSummaryRequest $request)
    {
        $driver = auth()->user();
        $validated = $request->validated();
       $query = SummaryDriverPayment::with('driver')
        ->where('driver_id', $driver->id)
        ->select('id', 'driver_id', 'invoice_number', 'total_amount_usd', 'total_amount_khr', 'payment_date')
         ->when(Arr::get($validated, 'date'), function ($query, $date) {
                $dates = explode(' to ', $date);

                if (count($dates) === 2) {
                    $from = $dates[0].' 00:00:00';
                    $to = $dates[1].' 23:59:59';

                    return $query->whereBetween('payment_date', [$from, $to]);
                } elseif (count($dates) === 1 && $dates[0]) {
                    return $query->whereBetween('payment_date', [
                        $dates[0].' 00:00:00',
                        $dates[0].' 23:59:59',
                    ]);
                }

                return $query;
            })
        ->orderBy('created_at', 'desc');
        $perPage = $request->input('per_page', 10);
        $data = $query->paginate($perPage);

        return $this->successResponse($data, 'Driver invoice summary');
    }

    /**
     * invoice details
     */
    public function invoiceDetail($invoice){
        $summary = SummaryDriverPayment::with(['confirmPayments', 'driverPayments.delivery', 'driver'])
            ->where('invoice_number',$invoice)->first();

        $deliveryIds = $summary->driverPayments->where('status', 'collected')
            ->pluck('delivery_id')->toArray();

        $deliveries = Delivery::with(['payments', 'customer'])
        ->whereIn('id', $deliveryIds)->get()
        ->map(function ($delivery) {
            return [
                'code' => $delivery->code,
                'assigned_at'   => $delivery->assigned_at,
                'receiver_address' => $delivery->receiver_address,
                'receiver_phone' => $delivery->receiver_phone,
                'original_amount_usd' => $delivery->amount_usd,
                'original_amount_khr' => $delivery->amount_khr,
                'completed_at'=>$delivery->completed_at,
                'note'=>$delivery->note,
                'customer' => [
                    'name'=>$delivery->customer?->customer_name,
                    'ref_code'=>$delivery->customer?->ref_code,
                    'phone'=>$delivery->customer?->phone,
                    'type_of_customer'=>$delivery->customer?->type_of_customer,
                ],
                'collect_payments'=>[
                   'amount_usd' => $delivery->payments
                        ->where('currency', 'usd')
                        ->sum('amount'),

                    'amount_khr' => $delivery->payments
                        ->where('currency', 'khr')
                        ->sum('amount'),
                ],
            ];
        });

        $invoiceSummary = [
            'invoice_number' => $summary->invoice_number,
            'exchange_rate' => $summary->exchange_rate,
            'total_amount_usd' => $summary->total_amount_usd,
            'total_amount_khr' => $summary->total_amount_khr,
            'completed' => $summary->completed,
            'date' => $summary->payment_date,
            'driver' => $summary->driver->display_name,
        ];

       

        return $this->successResponse(['summary'=>$invoiceSummary,'deliveries'=>$deliveries], 'Invoice listing');
  
    }


}
