<?php

namespace App\Http\Controllers;

use App\Enums\Status;
use App\Models\AssignZone;
use App\Models\User;
use App\Models\ZoneType;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class AssignZoneController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        // $this->middleware('permission:assign-zone.view')->only(['index', 'show']);
        // $this->middleware('permission:assign-zone.create')->only(['create', 'store']);
        // $this->middleware('permission:assign-zone.edit')->only(['edit', 'update']);
        // $this->middleware('permission:assign-zone.delete')->only(['destroy']);
    }

    public function index(Request $request)
    {
        // dd($request->all());
        if ($request->ajax()) {
            $branchId = auth()->user()->default_branch_id;
            $data = AssignZone::with(['zonePrices.zoneType', 'driver'])
                ->where('branch_id', $branchId)
                ->orderBy('driver_id', 'desc');

            return DataTables::of($data)
                ->addColumn('driver_name', function ($row) {
                    return $row->driver->display_name ?? '-';
                })
                ->addColumn('zone_type_zones', function ($row) {
                    return $row->zonePrices
                        ->groupBy(fn ($z) => $z->zoneType->zone_type ?? 'Unknown')
                        ->map(function ($zones, $type) {
                            $zoneNames = $zones->pluck('zone_name')->join(', ');

                            return "<strong>{$type}</strong>: {$zoneNames}";
                        })
                        ->implode('<br>');
                })
                ->addColumn('status', function ($row) {
                    return $row->status ?? 'inactive';
                })
                ->addColumn('action', function ($row) {
                    return view('pages.configuration.maping-zone.action-dropdown', compact('row'))->render();
                })
                ->rawColumns(['zone_type_zones', 'action']) // allow HTML
                ->make(true);
        }

        return view('pages.configuration.maping-zone.index');
    }

    public function create()
    {
        $deliveryBy = User::pluck('display_name', 'id');
        $zoneTypes = ZoneType::with('zonePrices')
            ->where('status', 'active') // adjust your status check
            ->get();

        return view('pages.configuration.maping-zone.edit', [
            'deliveryBy' => $deliveryBy,
            'zoneTypes' => $zoneTypes,
            'assingZone' => null,
            'assingZones' => collect(),
        ]);
    }

    // Show form for editing assignment by driver id
    public function edit($driverId)
    {
        $assingZones = AssignZone::where('driver_id', $driverId)
            ->with('zonePrices')
            ->get();

        if ($assingZones->isEmpty()) {
            abort(404, "AssignZone not found for driver id {$driverId}");
        }

        $assingZone = $assingZones->first();

        $deliveryBy = User::pluck('display_name', 'id');
        $zoneTypes = ZoneType::with('zonePrices')
            ->where('status', 'active') // adjust your status check
            ->get();

        return view('pages.configuration.maping-zone.edit', [
            'deliveryBy' => $deliveryBy,
            'zoneTypes' => $zoneTypes,
            'assingZone' => $assingZone,
            'assingZones' => $assingZones,
        ]);
    }

    // Store new assignment
    public function store(Request $request)
    {
        // dd($request->all());
        $validated = $request->validate([
            'delivery_by' => ['required', 'exists:users,id'],
            'zone' => ['required', 'array'],
            'zone.*' => ['exists:zone_price,id'],
        ]);

        // Prevent zones assigned to other drivers
        $assignedZoneIds = AssignZone::where('driver_id', '!=', $validated['delivery_by'])
            ->pluck('zone_price_id')
            ->toArray();

        foreach ($validated['zone'] as $zoneId) {
            if (in_array($zoneId, $assignedZoneIds)) {
                return back()->withErrors(['zone' => 'One or more zones are already assigned to other drivers.'])->withInput();
            }
        }

        // Save assignments
        foreach ($validated['zone'] as $zoneId) {
            AssignZone::create([
                'driver_id' => $validated['delivery_by'],
                'zone_price_id' => $zoneId,
                'branch_id' => auth()->user()->default_branch_id,
                'created_by' => auth()->user()->id,
                'status' => Status::ACTIVE->value,
            ]);
        }

        return redirect()->route('assign-zones.index')->with('success', 'Assigned zones saved successfully.');
    }

    // Update assignment for a driver
    public function update(Request $request, $driverId)
    {
        $validated = $request->validate([
            'delivery_by' => ['required', 'exists:users,id'],
            'zone' => ['required', 'array'],
            'zone.*' => ['exists:zone_price,id'],
        ]);

        AssignZone::where('driver_id', $driverId)->delete();

        $assignedZoneIds = AssignZone::where('driver_id', '!=', $validated['delivery_by'])
            ->pluck('zone_price_id')
            ->toArray();

        foreach ($validated['zone'] as $zoneId) {
            if (in_array($zoneId, $assignedZoneIds)) {
                return back()->withErrors(['zone' => 'One or more zones are already assigned to other drivers.'])->withInput();
            }
        }

        foreach ($validated['zone'] as $zoneId) {
            AssignZone::create([
                'driver_id' => $validated['delivery_by'],
                'zone_price_id' => $zoneId,
                'branch_id' => auth()->user()->default_branch_id,
                'created_by' => auth()->user()->id,
                'updated_by' => auth()->user()->id,
                'status' => Status::ACTIVE->value,
            ]);
        }

        return redirect()->route('assign-zones.index')->with('success', 'Assigned zones updated successfully.');
    }
}
