<?php

namespace App\Http\Controllers;

use App\Enums\Status;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    public function loginView()
    {
        return view('login');
    }

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'login' => ['required'],
            'password' => ['required'],
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $validated = $validator->validated();

        $key = 'login1:'.$request->ip();

        // Rate limit
        if (RateLimiter::tooManyAttempts($key, 3)) {
            $seconds = RateLimiter::availableIn($key);
            $minutes = ceil($seconds / 60);

            return redirect()->back()->withErrors([
                'login' => "Too many login attempts. Try again in $minutes minute(s).",
            ])->withInput();
        }

        // 🔥 Allow login by name OR phone
        $user = \App\Models\User::where('name', $validated['login'])
            ->orWhere('phone', $validated['login'])
            ->first();

        if (
            ! $user ||
            $user->status !== Status::ACTIVE->value ||
            ! \Hash::check($validated['password'], $user->password)
        ) {
            RateLimiter::hit($key, 900);

            return redirect()->back()->withErrors([
                'login' => 'The credentials do not match our records or your account is inactive.',
            ])->withInput();
        }

        // Login success
        RateLimiter::clear($key);
        \Auth::login($user);
        $request->session()->regenerate();

        session(['current_branch_id' => $user->default_branch_id]);

        return redirect()->route('dashboards.index');
    }

    public function logout()
    {
        auth()->logout();

        return redirect()->route('login');
    }
}
