<?php

namespace App\Http\Controllers;

use App\Enums\BookingStatus;
use App\Enums\Status;
use App\Http\Requests\Booking\AssignPickUpRequest;
use App\Http\Requests\Booking\StatusRequest;
use App\Http\Requests\BookingOrder\CreateRequest;
use App\Http\Requests\Customers\BookPickupRequest;
use App\Models\BookPickUp;
use App\Models\Customer;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\Storage;
class BookingPickUpController extends Controller
{
      /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $user = $request->user();
            $branchId = auth()->user()->default_branch_id;
            $data = BookPickUp::with('customer','images','pickupBy')->where('branch_id', $branchId)
                ->when($user->hasRole('customer'), function ($query) use ($user) {
                    $query->where('customer_id', $user->customer_id);
                })
                ->when($request->entry_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->entry_at);
                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('date', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        return $query->whereBetween('date', [
                            $dates[0].' 00:00:00',
                            $dates[0].' 23:59:59',
                        ]);
                    }

                    return $query;
                })
                ->orderBy('created_at', 'desc');
           
            return DataTables::of($data)
                 ->addIndexColumn()
                ->addColumn('customer_name', fn ($row) => $row->customer->customer_name ?? 'N/A')
                ->addColumn('phone', fn ($row) => $row->customer->phone ?? 'N/A')
                ->addColumn('url_map', fn ($row) => $row->customer->url_map ?? 'N/A')
                ->editColumn('date', function ($row) {
                    if (! $row->date) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->created_at)->format('d/m/Y h:i A');
                })
            
                ->addColumn('images', function ($row) {
                if ($row->images->isEmpty()) return '<span class="text-slate-400">No Image</span>';

                    $count = $row->images->count();
                    $html = '<div class="flex gap-1">';

                    foreach ($row->images->take(4) as $index => $image) {
                        $html .= '
                            <img src="'.asset("storage/".$image->image).'" 
                                class="h-10 w-10 rounded object-cover border cursor-pointer"
                                data-images=\''.json_encode($row->images->pluck('image')).'\'
                                data-index="'.$index.'"
                            >';
                    }

                    if ($count > 4) {
                        $html .= '<div class="h-10 w-10 flex items-center justify-center rounded bg-black text-white text-xs cursor-pointer">
                                    +'.($count - 4).'
                                </div>';
                    }

                    $html .= '</div>';

                    return $html;
                })
                ->addColumn('pickup', fn ($row) => $row->pickupBy->display_name ?? 'N/A')
                ->rawColumns(['pickup','images'])
                ->make(true);
        }
        
        $deliveryBy = User::select('id','display_name')
        ->where('status', Status::ACTIVE->value)
        ->get();

        return view('pages.customers.pickup.index', compact('deliveryBy'));
    }


   // Display Create Form
    public function create()
    {
        $branchId = auth()->user()->default_branch_id;

        if (auth()->user()->hasRole('customer')) {
            $customer = Customer::active()
                ->where('id', auth()->user()->customer_id)
                ->get();
        } else {
            $customer = Customer::active()
                ->where('branch_id', $branchId)
                ->get();
        }

        return view('pages.customers.pickup.create', compact('customer'));
    }

     /**
     * Store a newly created resource in storage.
     */
    public function store(BookPickupRequest $request)
        {
            $item = $request->validated();

            DB::beginTransaction();

            try {
                $pickup = BookPickUp::create([
                    'customer_id'   => $item['customer_id'],
                    'branch_id'     => auth()->user()->default_branch_id,
                    'total_package' => $item['total_package'],
                    'status'        => BookingStatus::BOOKING->value,
                    'note'          => $item['note'] ?? null,
                    'date'          => now(),
                ]);

                if ($request->hasFile('images')) {
                    foreach ($request->file('images') as $image) {
                        $path = $image->store('book-pickup', 'public');

                        $pickup->images()->create([
                            'image' => $path,
                        ]);
                    }
                }


                $image = $pickup->images->first();

                $imagePath = $image
                    ? storage_path('app/public/' . $image->image) // local path
                    : null;


                sendTelegramBookPickUpNotification([
                    'phone' => $pickup->customer->phone,
                    'customer' => $pickup->customer->customer_name,
                    'date' => $pickup->date,
                    'total_package' => $pickup->total_package,
                    'map_url' => $pickup->customer->url_map,
                    'images' =>$imagePath,
                    'remark'=>$pickup->note,
                ]);


                DB::commit();

                return redirect()
                    ->route('book-pickup.index')
                    ->with('success', 'Book pickup successfully.');

            } catch (\Exception $e) {
                DB::rollBack();
                return back()->with('error','Failed to booking pickup.');
            }
        }

    

    public function processAssignPickup(AssignPickUpRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            // Load pickup with customer and images
            $pickup = BookPickUp::with('customer', 'images','pickupBy')->findOrFail($data['book_pickup_id']);
            $pickup->status = 'pickup';
            $pickup->pickup_by = $data['pickup_by'];
            $pickup->note = $data['note'];
            $pickup->save();

            $image = $pickup->images->first();

            $imagePath = $image
                ? storage_path('app/public/' . $image->image) // local path
                : null;


            sendTelegramPickUpNotification([
                'phone' => $pickup->customer->phone,
                'customer' => $pickup->customer->customer_name,
                'date' => $pickup->date,
                'total_package' => $pickup->total_package,
                'map_url' => $pickup->customer->url_map,
                'images' =>$imagePath,
                'pickup_by'=>$pickup->pickupBy?->display_name,
            ]);


    
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Assign driver to pickup success.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('updated booking status error: '.$e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong.',
            ], 500);
        }
    }


}
