<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Http\Requests\Delivery\ChangeDriverRequest;
use App\Models\Delivery;
use App\Models\DeliveryStatusHistory;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class ChangeDriverController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:delivery.transfer')->only(['index', 'changeDriver']);
        $this->middleware('permission:delivery.transfer-create')->only(['store']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Delivery::where('branch_id', auth()->user()->default_branch_id)
                ->whereIn('status', [DeliveryStatus::ASSIGNED->value,DeliveryStatus::CANCELED->value])
                ->when($request->delivery_by, function ($query, $delivery_by) {
                    return $query->where('delivery_by', $delivery_by);
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })

                ->addColumn('delivery_by', function ($row) {
                    return $row->deliveryBy->display_name ?? 'N/A';
                })

                ->editColumn('pickup_at', function ($row) {
                    if (! $row->pickup_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->pickup_at)->format('d/m/Y h:i A');
                })

                ->addColumn('status_label', function ($row) {
                    $status = DeliveryStatus::tryFrom($row->status);

                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                        DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                        DeliveryStatus::ASSIGNED => 'text-info',
                        DeliveryStatus::DELIVERED => 'text-success',
                        DeliveryStatus::COMPLETED => 'text-success',
                        DeliveryStatus::RETURNED => 'text-warning',
                        DeliveryStatus::CANCELED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$label}</span>
                        </div>
                        HTML;
                })

                ->rawColumns(['status_label'])
                ->make(true);

        }

        return view('pages.delivery.change-driver');
    }

    public function changeDriver()
    {
        $deliveryBy = User::get()->pluck('display_name', 'id');

        return view('pages.delivery.change-driver', compact('deliveryBy'));
    }

    public function store(ChangeDriverRequest $request)
{
    $data = $request->validated();

    DB::beginTransaction();

    try {
        $deliveryIds = $data['delivery_ids'];
        $deliveryById = $data['new_delivery_by'];

        // Determine values based on $deliveryById
        if ($deliveryById === 0 || $deliveryById === '0') {
            // Assign back to stock
            $status = DeliveryStatus::IN_STOCK->value;
            $deliveryById = null; // no driver
            $cancelId = null;
            $note = 'Assigned back to stock';
        } else {
            // Assign to new driver
            $status = DeliveryStatus::ASSIGNED->value;
            $cancelId = null; // optional, keep existing if needed
            $note = 'Assigned to new driver';
        }

        // Update deliveries
        Delivery::whereIn('id', $deliveryIds)
            ->update([
                'delivery_by' => $deliveryById,
                'status' => $status,
                'cancel_reason_id' => $cancelId,
                'updated_by' => auth()->id(),
            ]);

        // Insert history
        DeliveryStatusHistory::insert(
            array_map(function ($deliveryId) use ($status, $note) {
                $now = now();
                return [
                    'delivery_id' => $deliveryId,
                    'status' => $status,
                    'changed_by' => auth()->id(),
                    'changed_at' => $now,
                    'note' => $note,
                    'created_at' => $now,
                    'updated_at' => $now,
                ];
            }, $deliveryIds)
        );

        DB::commit();

        return back()->with('success', 'Change driver successfully.');

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('Failed to change driver: '.$e->getMessage());

        return redirect()->back()->withInput()->with('error', 'Something when wrong!.');
    }
}


    public function store1(ChangeDriverRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $deliveryIds = $data['delivery_ids'];
            $deliveryById = $data['new_delivery_by'];

            Delivery::whereIn('id', $deliveryIds)
                ->where('status', DeliveryStatus::ASSIGNED->value)
                ->update([
                    'delivery_by' => $deliveryById,
                    'status' => DeliveryStatus::ASSIGNED->value,
                    'updated_by' => auth()->id(),
                ]);

            DeliveryStatusHistory::insert(
                array_map(function ($deliveryId) {
                    $now = now();

                    return [
                        'delivery_id' => $deliveryId,
                        'status' => DeliveryStatus::ASSIGNED->value,
                        'changed_by' => auth()->id(),
                        'changed_at' => $now,
                        'note' => 'Assigned to new driver',
                        'created_at' => $now,
                        'updated_at' => $now,
                    ];
                }, $deliveryIds)
            );

            DB::commit();

            return back()->with('success', 'Change driver successfully.');

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Failed to change driver: '.$e->getMessage());

            return redirect()->back()->withInput()->with('error', 'Failed to change driver. Please try again.');
        }
    }
}
