<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Http\Requests\DataInquiry\CreateRequest;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryStatusHistory;
use App\Models\User;
use App\Models\ZonePrice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class DataInquiryController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:delivery.view')->only(['index', 'show']);
        $this->middleware('permission:delivery.create')->only(['create', 'store']);
        $this->middleware('permission:delivery.edit')->only(['edit', 'update']);
        $this->middleware('permission:delivery.delete')->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $currentBranch = auth()->user()->default_branch_id;
        $driver = User::get()->pluck('display_name', 'id');
        $customer = Customer::active()
            ->where('branch_id', $currentBranch)
            ->select('id', 'customer_name', 'phone', 'currency', 'type_of_customer', 'mou_price')
            ->get();

        $zone = ZonePrice::where('status', 'active')
            ->where('branch_id', $currentBranch)
            ->select('id', 'zone_name', 'price', 'currency')
            ->get();

        return view('pages.delivery.inquiry.create',
            compact('driver', 'customer', 'zone')
        );
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

   /**
     * Store a newly created resource in storage.
     */
    public function store(CreateRequest $request)
    {
        DB::beginTransaction();

        try {
            $lastNumber = Delivery::getLastCodeNumber();
            $createdDeliveryIds = [];

            foreach ($request->items as $item) {
                $lastNumber++;

                $code = 'T' . str_pad($lastNumber, 7, '0', STR_PAD_LEFT);

                $status = $request->filled('delivery_by')
                    ? DeliveryStatus::ASSIGNED->value
                    : DeliveryStatus::IN_STOCK->value;

                // Create delivery
                $delivery = Delivery::create([
                    'branch_id'        => auth()->user()->default_branch_id,
                    'code'             => $code,
                    'customer_id'      => $request->customer_id,
                    'zone_price_id'    => $item['zone_id'],
                    'receiver_address' => $item['receiver_address'],
                    'receiver_phone'   => $item['receiver_phone'],
                    'delivery_price'   => $item['delivery_price'],
                    'pickup_by'        => $request->pickup_by,
                    'amount_usd'       => $item['amount_usd'] ?? 0,
                    'amount_khr'       => $item['amount_khr'] ?? 0,
                    'type_of_delivery' => 'Normal',
                    'delivery_by'      => $request->delivery_by,
                    'status'           => $status,
                    'created_by'       => auth()->id(),
                    'pickup_at'        => now(),
                ]);

                // Prepare status history
                $statuses = [
                    [
                        'delivery_id' => $delivery->id,
                        'status'      => DeliveryStatus::IN_STOCK->value,
                        'changed_by'  => auth()->id(),
                        'note'        => 'Batch created',
                        'created_at'  => now(),
                        'updated_at'  => now(),
                    ],
                ];

                if ($request->filled('delivery_by')) {
                    $statuses[] = [
                        'delivery_id' => $delivery->id,
                        'status'      => DeliveryStatus::ASSIGNED->value,
                        'changed_by'  => auth()->id(),
                        'note'        => 'Assigned to driver',
                        'created_at'  => now(),
                        'updated_at'  => now(),
                    ];
                }

                // Insert status history
                DeliveryStatusHistory::insert($statuses);

                $createdDeliveryIds[] = $delivery->id;
            }

            DB::commit();

            // Redirect to print
            return count($createdDeliveryIds) === 1
                ? redirect()->route('inquiry.invoice.print', $createdDeliveryIds[0])
                : redirect()->route(
                    'inquiry.invoice.batch',
                    ['ids' => implode(',', $createdDeliveryIds)]
                );

        } catch (\Throwable $e) {
            DB::rollBack();
            report($e);

            return back()->with('error', 'Failed to create delivery batch.');
        }
    }


    public function inquiryPrintInvoice($id)
    {
        $delivery = Delivery::select([
                'id',
                'code',
                'customer_id',
                'receiver_address',
                'receiver_phone',
                'delivery_price',
                'amount_usd',
                'amount_khr',
                'created_at',
            ])
            ->with('customer:id,customer_name,phone')
            ->findOrFail($id);

        return view('pages.delivery.inquiry.print-invoice', compact('delivery'));
    }


    public function inquiryPrintBatchInvoice($ids)
    {
        $ids = explode(',', $ids);

        $deliveries = Delivery::select([
                'id',
                'code',
                'customer_id',
                'receiver_address',
                'receiver_phone',
                'delivery_price',
                'amount_usd',
                'amount_khr',
                'created_at',
            ])
            ->with('customer:id,customer_name')
            ->whereIn('id', $ids)
            ->orderBy('id')
            ->get();

        return view('pages.delivery.inquiry.print-batch-invoice', compact('deliveries'));
    }


    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
