<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\SummaryDriverPayment;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class DriverPaymentController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:driver-payment.view')->only(['index']);
    }

    public function index(Request $request)
    {
        $user = $request->user();
        if ($request->ajax()) {

            $data = SummaryDriverPayment::with(['confirmPayments', 'driver', 'createdBy'])
                ->when($user->hasRole('driver'), function ($query) use ($user) {
                    $query->where('driver_id', $user->id);
                })
                ->when($request->search['value'] ?? null, function ($query, $search) {
                    $query->where('invoice_number', 'LIKE', "%{$search}%");
                })
                ->when($request->delivery_by, function ($query, $delivery_by) {
                    return $query->where('driver_id', $delivery_by);
                })
                ->when($request->entry_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->entry_at);
                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('payment_date', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        return $query->whereBetween('payment_date', [
                            $dates[0].' 00:00:00',
                            $dates[0].' 23:59:59',
                        ]);
                    }

                    return $query;
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('driver', function ($row) {
                    $driver = $row->driver;
                    $avatar = 'https://cdn-icons-png.flaticon.com/512/149/149071.png';
                    $name = $driver->display_name ?? '-';
                    $phone = $driver->phone ?? '-';
                    $date = $row->payment_date ? Carbon::parse($row->payment_date)->format('d/m/Y h:i A') : '-';

                    return '
                        <div class="flex items-center space-x-3">
                            <img class="w-10 h-10 rounded-full object-cover" src="'.$avatar.'" alt="'.$name.'">
                            <div class="flex flex-col">
                                <span class="font-medium text-slate-700 dark:text-navy-100">'.$name.'</span>
                                <span class="text-xs text-slate-400 dark:text-navy-300">'.$phone.'</span>
                                <span class="text-xs text-slate-400 dark:text-navy-300">'.$date.'</span>
                            </div>
                        </div>
                    ';
                })

                ->addColumn('collect_from_driver', function ($row) {
                    // USD
                    $usdAmount = $row->confirmPayments->where('currency', 'usd')->sum('amount');
                    $usdMethod = optional($row->confirmPayments->where('currency', 'usd')->first())->payment_method ?? 'N/A';

                    // KHR
                    $khrAmount = $row->confirmPayments->where('currency', 'khr')->sum('amount');
                    $khrMethod = optional($row->confirmPayments->where('currency', 'khr')->first())->payment_method ?? 'N/A';

                    if ($row->driver_collect_usd != $usdAmount) {
                        $text = 'text-red-500';
                    } else {
                        $text = '';
                    }

                    return '
                        <div class="flex flex-col text-sm">
                         <a class="text-success">Collected From Driver</a>
                            <ul class="list-disc list-inside mt-1 text-slate-600 dark:text-navy-200">
                                <li class="'.$text.'">'.number_format($usdAmount, 2).' USD ('.ucfirst($usdMethod).')</li>
                                <li>'.number_format($khrAmount, 2).' KHR ('.ucfirst($khrMethod).')</li>
                            <ul>
                        </div>
                    ';
                })

                ->addColumn('driver_collect_from_customer', function ($row) {
                    $usdAmount = $row->confirmPayments->where('currency', 'usd')->sum('amount');
                    if ($row->driver_collect_usd != $usdAmount) {
                        $text = 'text-red-500';
                    } else {
                        $text = '';
                    }

                    return '
                        <div class="flex flex-col text-sm">
                         <a class="text-success">Amount POD</a>
                            <ul class="list-disc list-inside mt-1 text-slate-600 dark:text-navy-200">
                                <li class="'.$text.'">'.number_format($row->driver_collect_usd, 2).' USD</li>
                                <li>'.number_format($row->driver_collect_khr, 2).' KHR</li>
                            <ul>
                        </div>
                    ';
                })

                ->addColumn('total_package', function ($row) {
                    return '
                        <ul class="text-sm">
                            <li class="text-primary">🚚 Assigned: '.$row->assigned.'</li>
                             <li class="text-success">✔ Completed: '.$row->completed.'</li>
                            <li class="text-error">✖ Canceled: '.$row->canceled.'</li>
                        </ul>
                    ';
                })

                ->addColumn('action', function ($row) {
                    return '
                        <div class="flex flex-col text-sm">
                             <strong class="text-center">'.($row->createdBy->display_name ?? 'N/A').'</strong>
                             <hr/>
                             <div class="mt-2 badge p-2 space-x-2 bg-blue-100 text-blue-800 dark:bg-blue-500 dark:text-white"><strong><a href="'.route('driver-collect-payment.edit', $row->id).'">Edit & View</a></strong></div>
                        </div>
                    ';
                })
                ->addColumn('invoice', function ($row) {
                    return '
                        <div class="flex flex-col text-sm">
                            <a class="text-success" href="'.route('driver-collect-payment.edit', $row->id).'">'.$row->invoice_number.'</a>
                        </div>
                    ';
                })
                ->rawColumns(['action', 'driver', 'collect_from_driver', 'driver_collect_from_customer', 'total_package', 'invoice'])
                ->make(true);

        }

        if (!auth()->user()->hasRole('customer')) {
            $deliveryBy = User::where('status', 'active')
                ->whereHas('roles', function ($query) {
                    $query->where('name', 'driver');
                })
                ->pluck('display_name', 'id');
        } else {
            $deliveryBy = collect(); // empty collection
        }
            return view('pages.collect.index', compact('deliveryBy'));
        }
}
