<?php

namespace App\Http\Controllers;

use App\Enums\BookingStatus;
use App\Http\Requests\Booking\InTransitStatusRequest;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryStatusHistory;
use App\Models\InTransit;
use App\Models\InTransitSummary;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class InTransitController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $branchId = $request->user()->default_branch_id;
        $warehouse = Warehouse::get()->pluck('name', 'id');
        $customer = Customer::active()
            ->where('branch_id', $branchId)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();

        $statusOptions = collect(BookingStatus::cases())
            ->filter(fn ($status) => $status->value === 'arrived_pp')
            ->map(function ($status) {
                return [
                    'value' => $status->value,
                    'label' => $status->label(),
                ];
            });

        return view('pages.booking.in-transit', compact('warehouse', 'customer', 'statusOptions'));
    }

    public function inTransitAjax(Request $request)
    {
        if ($request->ajax()) {
            $data = InTransitSummary::when($request->from_warehouse, function ($query, $from_warehouse) {
                return $query->where('from_warehouse_id', $from_warehouse);
            })
                ->when($request->to_warehouse, function ($query, $to_warehouse) {
                    return $query->where('to_warehouse_id', $to_warehouse);
                })
                ->when($request->shipping_method, function ($query, $shipping_method) {
                    return $query->where('shipping_method', $shipping_method);
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)

                ->addColumn('from_warehouse_name', function ($row) {
                    return $row->fromWarehouse->name ?? 'N/A';
                })
                ->addColumn('to_warehouse_name', function ($row) {
                    return $row->toWarehouse->name ?? 'N/A';
                })
                ->addColumn('truck_by', function ($row) {
                    return $row->truckBy->display_name ?? 'N/A';
                })
                ->addColumn('created_by', function ($row) {
                    return $row->createdBy->display_name ?? 'N/A';
                })
                ->editColumn('created_at', function ($row) {
                    if (! $row->created_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->created_at)->format('d/m/Y h:i A');
                })
                ->addColumn('status_label', function ($row) {
                    $status = BookingStatus::tryFrom($row->status);
                    $label = $status?->label() ?? 'Completed';
                    $colorClass = match ($status) {
                        BookingStatus::IN_TRANSIT => 'text-info',
                        default => 'text-success',
                    };

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$label}</span>
                        </div>
                    HTML;
                })

                ->addColumn('action', function ($row) {
                    return view('pages.booking.action-dropdown-in-transit', [
                        'row' => $row,
                        'canEdit' => $row->is_paid !== true,
                    ])->render();
                })

                ->rawColumns(['action', 'status_label'])
                ->make(true);

        }
    }

    /** printInvoice */
    public function printInvoiceInTransit(string $invoice)
    {
        $summary = InTransitSummary::where('invoice_number', $invoice)
            ->firstOrFail();

        $deliveries = InTransit::with([
            'delivery.fromWarehouse',
            'delivery.toWarehouse',
        ])
            ->where('in_transit_summary_id', $summary->id)
            ->get();

        $car = User::findOrFail($summary->truck_id);

        return view('pages.booking.in-transit-invoice', compact(
            'summary',
            'deliveries',
            'car'
        ));
    }

    /* ajx */
    public function updateStatusArrived(InTransitStatusRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $summary = InTransitSummary::where('invoice_number', $data['delivery_id'])->first();
            $transit = InTransit::where('in_transit_summary_id', $summary->id)->get();
            foreach ($transit as $v) {
                $delivery = Delivery::findOrFail($v->delivery_id);
                $delivery->status = $data['status'];
                $delivery->note = $data['note'];
                $delivery->save();

                DeliveryStatusHistory::create([
                    'delivery_id' => $v->delivery_id,
                    'status' => BookingStatus::ARRIVEDWAREHOUSEPP->value,
                    'note' => $delivery->note ?? BookingStatus::ARRIVEDWAREHOUSEPP->value,
                    'changed_by' => auth()->user()->id,
                    'changed_at' => now(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            $summary->status = 'completed';
            $summary->updated_by = auth()->user()->id;
            $summary->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Update status successfully.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('updated arrived pp status error: '.$e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while processing. Please try again later.',
            ], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
