<?php

namespace App\Http\Controllers;

use App\Http\Requests\Roles\CreateRequest;
use App\Http\Requests\Roles\UpdateRequest;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Yajra\DataTables\Facades\DataTables;

class RoleController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:roles.view')->only(['index', 'show']);
        $this->middleware('permission:roles.create')->only(['create', 'store']);
        $this->middleware('permission:roles.edit')->only(['edit', 'update']);
        $this->middleware('permission:roles.delete')->only(['destroy']);
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $query = Role::with('permissions')->select('roles.*');

            return DataTables::of($query)
                ->addColumn('permissions', function ($role) {
                    return $role->permissions->pluck('name')->implode(', ');
                })
                ->addColumn('action', function ($row) {
                    return view('pages.configuration.roles.action-dropdown', compact('row'))->render();
                })
                ->filter(function ($query) use ($request) {
                    if ($request->has('search') && $request->get('search')['value'] != '') {
                        $search = $request->get('search')['value'];
                        $query->where('name', 'like', "%{$search}%")
                            ->orWhereHas('permissions', function ($q) use ($search) {
                                $q->where('name', 'like', "%{$search}%");
                            });
                    }
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        return view('pages.configuration.roles.index');
    }

    public function create()
    {
        $permissions = Permission::all();

        return view('pages.configuration.roles.create', compact('permissions'));
    }

    public function store(CreateRequest $request)
    {
        $validate = $request->validated();
        $role = Role::create([
            'name' => $validate['name'],
            'guard_name' => 'web',
        ]);

        $permissions = Permission::whereIn('id', $validate['permissions'])->pluck('name')->toArray();

        $role->syncPermissions($permissions);

        return redirect()->route('roles.index')->with('success', 'Role created successfully.');
    }

    public function edit(Role $role)
    {
        $permissions = Permission::all();
        $rolePermissions = $role->permissions->pluck('id')->toArray();

        return view('pages.configuration.roles.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    public function update(UpdateRequest $request, Role $role)
    {
        $validated = $request->validated();
        $role->update([
            'name' => $validated['name'],
            'status' => $validated['status'],
        ]);

        $permissions = Permission::whereIn('id', $validated['permissions'])->pluck('name');
        $role->syncPermissions($permissions);

        return redirect()->route('roles.index')->with('success', 'Role updated successfully.');
    }

    public function destroy(Role $role)
    {
        $role->delete();

        return redirect()->route('roles.index')->with('success', 'Role deleted successfully.');
    }
}
