<?php

namespace App\Http\Controllers;

use App\Http\Requests\Users\CreateRequest;
use App\Http\Requests\Users\UpdateRequest;
use App\Models\Branch;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;
use Yajra\DataTables\Facades\DataTables;

class UserController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:users.view')->only(['index', 'show']);
        $this->middleware('permission:users.create')->only(['create', 'store']);
        $this->middleware('permission:users.edit')->only(['edit', 'update']);
        $this->middleware('permission:users.delete')->only(['destroy']);
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = User::with(['roles', 'branch'])
            ->when($request->branch, function ($query, $branch) {
                return $query->where('default_branch_id', $branch);
            })
            ->orderBy('created_at', 'desc');

            if ($request->filled('role')) {
                $role = $request->role;
                $data->whereHas('roles', function($q) use ($role) {
                    $q->where('name', $role);
                });
            }

            return DataTables::of($data)
            ->addIndexColumn()
                ->addColumn('avatar', function ($row) {
                    return '<img src="'.e($row->avatar_url).'" alt="Avatar" style="width:40px; height:40px; border-radius:50%;">';
                })
                ->addColumn('role', function ($row) {
                    return $row->roles->pluck('name')->join(', ');
                })
                ->addColumn('defaultBranch', function ($row) {
                    return $row->defaultBranch ? $row->defaultBranch->name : '-';
                })
                ->addColumn('action', function ($row) {
                    return view('pages.configuration.users.action-dropdown', compact('row'))->render();
                })
                ->rawColumns(['avatar', 'action'])
                ->make(true);
        }
 
        $branches = Branch::all();
        $roles = Role::all();

        return view('pages.configuration.users.index', compact('branches','roles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $branches = Branch::all();
        $roles = Role::all();

        return view('pages.configuration.users.create', compact('branches', 'roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CreateRequest $request)
    {
        $validated = $request->validated();
        $user = User::create([
            'display_name' => $validated['display_name'],
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'gender' => $validated['gender'],
            'status' => $validated['status'],
            'password' => Hash::make($validated['password']),
            'default_branch_id' => $validated['branches'][0],
        ]);

        $user->assignRole($validated['role']);
        $user->branches()->attach($validated['branches']);

        return redirect()->route('users.index')->with('success', 'User created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $user = User::with(['roles', 'branch'])->findOrFail($id);

        return view('pages.configuration.users.view', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $branches = Branch::all();
        $roles = Role::all();

        return view('pages.configuration.users.edit', compact('user', 'branches', 'roles'));
    }

    public function update(UpdateRequest $request, User $user)
    {
        $validated = $request->validated();

        $user->update([
            'display_name' => $validated['display_name'],
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'gender' => $validated['gender'],
            'status' => $validated['status'],
        ]);

        $user->syncRoles($validated['role']);
        $user->branches()->sync($validated['branches']);

        return redirect()->route('users.index')->with('success', 'User updated successfully.');
    }

    public function destroy(string $id)
    {
        $user = User::findOrFail($id);

        // Optional: detach roles before deleting
        $user->syncRoles([]);

        $user->delete();

        return redirect()->route('users.index')->with('success', 'User deleted successfully.');
    }
}
