<?php

namespace App\Http\Controllers;

use App\Models\Warehouse;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class WarehouseController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:branch.view')->only(['index', 'show']);
        $this->middleware('permission:branch.create')->only(['create', 'store']);
        $this->middleware('permission:branch.edit')->only(['edit', 'update']);
        $this->middleware('permission:branch.delete')->only(['destroy']);
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Warehouse::orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    return view('pages.configuration.warehouse.action-dropdown', compact('row'))->render();
                })
                ->make(true);
        }

        return view('pages.configuration.warehouse.index');
    }

    public function create()
    {
        return view('pages.configuration.warehouse.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'contact_name' => 'nullable|string|max:255',
            'contact_phone' => 'nullable|string|max:255',
            'status' => 'nullable|string|max:50',
            'city' => 'nullable|string|max:50',
            'address' => 'nullable|string',
            'postal' => 'nullable|string',
        ]);
        $validated['branch_id'] = auth()->user()->default_branch_id;
        Warehouse::create($validated);

        return redirect()->route('warehouse.index')->with('success', 'Warehouse created successfully.');
    }

    public function show(string $id)
    {
        $branch = Warehouse::findOrFail($id);

        return view('pages.configuration.warehouse.view', compact('branch'));
    }

    public function edit(string $id)
    {
        $branch = Warehouse::findOrFail($id);

        return view('pages.configuration.warehouse.edit', compact('branch'));
    }

    public function update(Request $request, string $id)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'contact_name' => 'nullable|string|max:255',
            'contact_phone' => 'nullable|string|max:255',
            'status' => 'nullable|string|max:50',
            'city' => 'nullable|string|max:50',
            'address' => 'nullable|string',
            'postal' => 'nullable|string',
        ]);

        $branch = Warehouse::findOrFail($id);
        $validated['branch_id'] = auth()->user()->default_branch_id;
        $branch->update($validated);

        return redirect()->route('warehouse.index')->with('success', 'warehouse updated successfully.');
    }

    public function destroy(string $id)
    {
        $branch = Warehouse::findOrFail($id);
        $branch->delete();

        return response()->json(['success' => 'warehouse deleted successfully.']);
    }
}
