<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Http\Requests\Delivery\AdjustmentDeliveryRequest;
use App\Http\Requests\DriverPayment\CreateRequest;
use App\Http\Requests\DriverPayment\StoreRequest;
use App\Http\Requests\DriverPayment\UpdateRequest;
use App\Models\ConfirmDriverPayment;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryPayment;
use App\Models\DriverPayment;
use App\Models\SummaryDriverPayment;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class DriverCollectPaymentController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:driver-payment.view')->only(['index', 'show']);
        $this->middleware('permission:driver-payment.create')->only(['create', 'store', 'driverCollectPayment']);
        $this->middleware('permission:driver-payment.edit')->only(['edit', 'update']);
        $this->middleware('permission:driver-payment.delete')->only(['destroy']);
        $this->middleware('permission:driver-payment.adjust')->only(['adjustDeliveryPayment']);
    }

    public function index(Request $request)
    {
        $user = $request->user();
        if ($request->ajax()) {
            $data = Delivery::where('branch_id', auth()->user()->default_branch_id)
                ->where('status', DeliveryStatus::COMPLETED->value)
                ->where('is_collected', false)

                ->when($user->hasRole('driver'), function ($query) use ($user) {
                    $query->where('delivery_by', $user->id);
                })

                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->when($request->delivery_by, function ($query, $delivery_by) {
                    return $query->where('delivery_by', $delivery_by);
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })
                ->addColumn('pickup_by', function ($row) {
                    return $row->pickupBy->display_name ?? 'N/A';
                })
                ->addColumn('delivery_by', function ($row) {
                    return $row->deliveryBy->display_name ?? 'N/A';
                })
                ->addColumn('created_by', function ($row) {
                    return $row->createdBy->display_name ?? 'N/A';
                })
                ->editColumn('created_at', function ($row) {
                    if (! $row->created_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->created_at)->format('d-F-Y');
                })
                ->addColumn('status_label', function ($row) {
                    $status = DeliveryStatus::tryFrom($row->status);

                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                        DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                        DeliveryStatus::ASSIGNED => 'text-info',
                        DeliveryStatus::DELIVERED => 'text-success',
                        DeliveryStatus::COMPLETED => 'text-success',
                        DeliveryStatus::RETURNED => 'text-warning',
                        DeliveryStatus::CANCELED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$label}</span>
                        </div>
                        HTML;
                })
                ->rawColumns(['status_label'])
                ->make(true);

        }

        $today = Carbon::today();

        $drivers = User::whereHas('deliveries', function ($query) use ($today) {
            $query->where('status', 'completed')
                ->whereDate('completed_at', $today);
        })->get();

        $deliveryBy = $drivers->pluck('display_name', 'id');

        $customer = Customer::active()
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->withCount(['deliveries as completed_today_count' => function ($query) use ($today) {
                $query->where('status', 'completed')
                    ->whereDate('completed_at', $today);
            }])
            ->having('completed_today_count', '>', 0)
            ->get();

        return view('pages.driver-collect-payment.index',
            compact('deliveryBy', 'customer')
        );
    }

    public function show(CreateRequest $request, $id)
    {
        $data = $request->validated();

        $deliveries = Delivery::with('deliveryBy', 'payments', 'customer')->where('delivery_by', $data['delivery_by'])
            ->whereIn('id', $data['delivery_ids'])
            ->get();

        if ($deliveries->isEmpty()) {
            return redirect()->back()->with('error', 'No deliveries found for the selected driver.');
        }

        $deliveryBy = User::findOrFail($data['delivery_by']);

        $statusCounts = Delivery::where('delivery_by', $data['delivery_by'])
            ->where('is_paid', 0)
            ->where('is_collected', 0)
            ->selectRaw('status, COUNT(*) as total')
            ->groupBy('status')
            ->pluck('total', 'status');

        return view('pages.driver-collect-payment.view-collect-payment', [
            'deliveries' => $deliveries,
            'deliveryBy' => $deliveryBy,
            'driverName' => $deliveryBy->name,
            'exchangeRate' => getExchangeRate(),
            'invoiceNumber' => 'INV-NOTYET',
            'statusCounts' => $statusCounts,
        ]);
    }

    public function store(StoreRequest $request)
    {

        $validated = $request->validated();

        try {
            DB::beginTransaction();

            $deliveryIds = $validated['delivery_ids'];

            $statusCounts = Delivery::where('delivery_by', $validated['delivery_by'])
                ->where('is_paid', 0)
                ->where('is_collected', 0)
                ->selectRaw('status, COUNT(*) as total')
                ->groupBy('status')
                ->pluck('total', 'status');

            // Create Payment Summary
            $summary = SummaryDriverPayment::create([
                'driver_id' => $validated['delivery_by'],
                'invoice_number' => 'INV-'.now()->format('YmdHis'),
                'exchange_rate' => getExchangeRate(),
                'total_amount_usd' => 0,
                'total_amount_khr' => 0,
                'driver_collect_usd' => $validated['driver_collect_usd'] ?? 0,
                'driver_collect_khr' => $validated['driver_collect_khr'] ?? 0,
                'payment_date' => now(),
                'created_by' => auth()->id(),
                'completed' => $statusCounts['completed'] ?? 0,
                'assigned' => $statusCounts['assigned'] ?? 0,
                'canceled' => $statusCounts['canceled'] ?? 0,
            ]);

            $totalUsd = 0;
            $totalKhr = 0;

            foreach ($request->payments as $payment) {
                ConfirmDriverPayment::create([
                    'summary_id' => $summary->id,
                    'driver_id' => $validated['delivery_by'],
                    'amount' => $payment['amount'],
                    'currency' => $payment['currency'],
                    'payment_method' => $payment['payment_method'],
                ]);
                if ($payment['currency'] === 'usd') {
                    $totalUsd += $payment['amount'];
                } else {
                    $totalKhr += $payment['amount'];
                }

            }

            // delivery complted
            foreach ($deliveryIds as $deliveryId) {
                DriverPayment::create([
                    'summary_id' => $summary->id,
                    'driver_id' => $validated['delivery_by'],
                    'delivery_id' => $deliveryId,
                    'status' => 'collected', // or reverted
                    'payment_date' => now(),
                    'note' => 'collected payment with driver',
                    'created_by' => auth()->id(),
                ]);
            }

            $assignCancel = Delivery::where('delivery_by', $validated['delivery_by'])
                ->where('is_paid', 0)
                ->where('is_collected', 0)
                ->whereIn('status', [
                    DeliveryStatus::ASSIGNED->value,
                    DeliveryStatus::CANCELED->value,
                ])
                ->select(['id', 'status'])
                ->get();

            // delivery assign & canceled
            foreach ($assignCancel as $del) {
                DriverPayment::create([
                    'summary_id' => $summary->id,
                    'driver_id' => $validated['delivery_by'],
                    'delivery_id' => $del->id,
                    'status' => $del->status,
                    'payment_date' => now(),
                    'note' => 'assigned & canceled',
                    'created_by' => auth()->id(),
                ]);
            }

            $summary->update([
                'total_amount_usd' => $totalUsd,
                'total_amount_khr' => $totalKhr,
            ]);

            Delivery::whereIn('id', $deliveryIds)->update([
                'is_collected' => 1,
                'collected_by' => auth()->id(),
            ]);

            DB::commit();

            return redirect()->route('driver-collect-payment.index')
                ->with('success', 'Driver payment recorded and delivery status updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Error processing payment: '.$e->getMessage());
        }
    }

    public function edit($id)
    {
        $summary = SummaryDriverPayment::with(['confirmPayments', 'driverPayments.delivery', 'driver'])
            ->findOrFail($id);

        $existingPayments = $summary->confirmPayments->map(function ($p) {
            return [
                'amount' => $p->amount,
                'currency' => $p->currency,
                'payment_method' => $p->payment_method,
            ];
        });

        $deliveryIds = $summary->driverPayments->where('status', 'collected')
            ->pluck('delivery_id')->toArray();

        $deliveries = Delivery::with(['payments', 'customer'])->whereIn('id', $deliveryIds)->get();

        return view('pages.driver-collect-payment.view-collect-payment', [
            'summary' => $summary,
            'existingPayments' => $existingPayments,
            'driverName' => $summary->driver->display_name,
            'exchangeRate' => $summary->exchange_rate,
            'invoiceNumber' => $summary->invoice_number,
            'deliveryBy' => $summary->driver,
            'deliveries' => $deliveries,
        ]);
    }

    public function update(UpdateRequest $request, $id)
    {
        $validated = $request->validated();

        $summary = SummaryDriverPayment::where('id', $id)->firstOrFail();

        try {
            DB::beginTransaction();

            // Update summary
            $summary->update([
                'driver_id' => $validated['delivery_by'],
                'exchange_rate' => getExchangeRate(),
                'payment_date' => now(),
                'updated_by' => auth()->id(),
                'driver_collect_usd' => $validated['driver_collect_usd'] ?? 0,
                'driver_collect_khr' => $validated['driver_collect_khr'] ?? 0,
            ]);

            $totalUsd = 0;
            $totalKhr = 0;
            // Delete existing payments
            ConfirmDriverPayment::where('summary_id', $summary->id)->delete();
            foreach ($request->payments as $payment) {
                ConfirmDriverPayment::create([
                    'summary_id' => $summary->id,
                    'driver_id' => $validated['delivery_by'],
                    'amount' => $payment['amount'],
                    'currency' => $payment['currency'],
                    'payment_method' => $payment['payment_method'],
                ]);

                if ($payment['currency'] === 'usd') {
                    $totalUsd += $payment['amount'];
                } else {
                    $totalKhr += $payment['amount'];
                }
            }

            // Delete & reinsert delivery links
            DriverPayment::where('summary_id', $summary->id)->where('status', 'collected')->delete();

            foreach ($request->delivery_ids as $deliveryId) {
                DriverPayment::create([
                    'summary_id' => $summary->id,
                    'driver_id' => $validated['delivery_by'],
                    'delivery_id' => $deliveryId,
                    'status' => 'collected',
                    'payment_date' => now(),
                    'note' => 'updated payment',
                    'created_by' => auth()->id(),
                ]);
            }

            $summary->update([
                'total_amount_usd' => $totalUsd,
                'total_amount_khr' => $totalKhr,
            ]);

            DB::commit();

            return redirect()->route('driver-payment.index')
                ->with('success', 'Driver payment updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Update failed: '.$e->getMessage());
        }
    }

    public function adjustDeliveryPayment(AdjustmentDeliveryRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $delivery = Delivery::findOrFail($data['delivery_id']);
            $delivery->delivery_price = $data['delivery_price'];
            $delivery->amount_usd = $data['amount_usd'];
            $delivery->amount_khr = $data['amount_khr'];
            $delivery->save();

            DeliveryPayment::where('delivery_id', $data['delivery_id'])->delete();

            foreach ($data['payments'] as $paymentTo) {
                DeliveryPayment::create([
                    'delivery_id' => $data['delivery_id'],
                    'payment_to' => $paymentTo['payment_to'],
                    'amount' => $paymentTo['amount'],
                    'currency' => $paymentTo['currency'],
                    'payment_method' => $paymentTo['payment_method'],
                    'note' => $data['note'] ?? null,
                    'created_by' => auth()->id(),
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Adjustment of delivery & payment was successful.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Payment error: '.$e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong. Please try again later.',
            ], 500);
        }
    }
}
