<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Http\Requests\Delivery\RemoveDeliveryRequest;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryStatusHistory;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class RemoveDeliveryController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:delivery.remove')->only(['index', 'removeDelivery']);
        $this->middleware('permission:delivery.remove-create')->only(['store']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Delivery::where('branch_id', auth()->user()->default_branch_id)
                ->whereIn('status', [
                    DeliveryStatus::IN_STOCK->value,
                    DeliveryStatus::CANCELED->value,
                    DeliveryStatus::RETURNED->value,
                    DeliveryStatus::ASSIGNED->value,
                ])
                ->when($request->delivery_by, function ($query, $delivery_by) {
                    return $query->where('delivery_by', $delivery_by);
                })
                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->where('is_active', 1)
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })

                ->addColumn('delivery_by', function ($row) {
                    return $row->deliveryBy->display_name ?? 'N/A';
                })

                ->editColumn('pickup_at', function ($row) {
                    if (! $row->pickup_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->pickup_at)->format('d-F-Y');
                })
                ->editColumn('note', function ($row) {
                    if (! $row->note) {
                        return 'N/A';
                    }

                    return $row->note;
                })

                ->addColumn('status_label', function ($row) {
                    $status = DeliveryStatus::tryFrom($row->status);

                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                        DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                        DeliveryStatus::ASSIGNED => 'text-info',
                        DeliveryStatus::DELIVERED => 'text-success',
                        DeliveryStatus::COMPLETED => 'text-success',
                        DeliveryStatus::RETURNED => 'text-warning',
                        DeliveryStatus::CANCELED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$label}</span>
                        </div>
                        HTML;
                })

                ->rawColumns(['status_label'])
                ->make(true);

        }

        return view('pages.delivery.remove-delivery');
    }

    public function removeDelivery()
    {
        $deliveryBy = User::get()->pluck('display_name', 'id');
        $customer = Customer::active()
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();

        return view('pages.delivery.remove-delivery', compact('deliveryBy', 'customer'));
    }

    public function store(RemoveDeliveryRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $deliveryIds = $data['delivery_ids'];
            $deletedReason = $data['deleted_reason'];

            Delivery::whereIn('id', $deliveryIds)
                ->whereNotIn('status', [DeliveryStatus::COMPLETED->value])
                ->update([
                    'status' => DeliveryStatus::CANCELED->value,
                    'deleted_by' => auth()->id(),
                    'is_active' => 0,
                    'deleted_reason' => $deletedReason,
                ]);

            DeliveryStatusHistory::insert(
                array_map(function ($deliveryId) use ($deletedReason) {
                    $now = now();

                    return [
                        'delivery_id' => $deliveryId,
                        'status' => DeliveryStatus::CANCELED->value,
                        'changed_by' => auth()->id(),
                        'changed_at' => $now,
                        'note' => $deletedReason.' #'.$deliveryId,
                        'created_at' => $now,
                        'updated_at' => $now,
                    ];
                }, $deliveryIds)
            );

            DB::commit();

            return back()->with('success', 'Deleted delivery successfully.');

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Failed to delete delivery: '.$e->getMessage());

            return redirect()->back()->withInput()->with('error', 'Failed to delete delivery. Please try again.');
        }
    }
}
