<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\User;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class ReportController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        return view('pages.reports.index');
    }

    public function driverEarning(Request $request)
    {
        if ($request->ajax()) {
            $data = Delivery::where('branch_id', auth()->user()->default_branch_id)
                ->where('status', DeliveryStatus::COMPLETED->value)
                ->when($request->delivery_by, function ($query, $delivery_by) {
                    return $query->where('delivery_by', $delivery_by);
                })
                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->when($request->status, function ($query, $status) {
                    return $query->where('status', $status);
                })
                ->when($request->completed_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->completed_at);

                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('completed_at', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        $date = $dates[0];

                        return $query->whereBetween('completed_at', [
                            $date.' 00:00:00',
                            $date.' 23:59:59',
                        ]);
                    }

                    return $query;
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })
                ->addColumn('pickup_by', function ($row) {
                    return $row->pickupBy->display_name ?? 'N/A';
                })
                ->addColumn('delivery_by', function ($row) {
                    return $row->deliveryBy->display_name ?? 'N/A';
                })

                ->editColumn('completed_at', function ($row) {
                    if (! $row->completed_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->completed_at)->format('d-F-Y');
                })
                ->addColumn('status_label', function ($row) {
                    $status = DeliveryStatus::tryFrom($row->status);
                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                        DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                        DeliveryStatus::ASSIGNED => 'text-info',
                        DeliveryStatus::DELIVERED => 'text-success',
                        DeliveryStatus::COMPLETED => 'text-success',
                        DeliveryStatus::RETURNED => 'text-warning',
                        DeliveryStatus::CANCELED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                            <div class="badge space-x-2.5 {$colorClass}">
                                <div class="size-2 rounded-full bg-current"></div>
                                <span>{$label}</span>
                            </div>
                        HTML;
                })
                ->addColumn('status_clickable', function ($row) {
                    return DeliveryStatus::tryFrom($row->status) === DeliveryStatus::ASSIGNED;
                })
                ->rawColumns(['status_label', 'status_clickable'])
                ->make(true);

        }

        $deliveryBy = User::get()->pluck('display_name', 'id');
        $customer = Customer::active()
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();
        $statusOptions = collect(DeliveryStatus::cases())->map(function ($status) {
            return [
                'value' => $status->value,
                'label' => $status->label(),
            ];
        });

        return view('pages.reports.driver-earning', compact('deliveryBy', 'customer', 'statusOptions'));
    }

    public function pickUpCommission(Request $request)
    {
        if ($request->ajax()) {
            $data = Delivery::where('branch_id', auth()->user()->default_branch_id)
                ->where('status', DeliveryStatus::COMPLETED->value)
                ->when($request->delivery_by, function ($query, $delivery_by) {
                    return $query->where('delivery_by', $delivery_by);
                })
                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->when($request->status, function ($query, $status) {
                    return $query->where('status', $status);
                })
                ->when($request->completed_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->completed_at);

                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('completed_at', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        $date = $dates[0];

                        return $query->whereBetween('completed_at', [
                            $date.' 00:00:00',
                            $date.' 23:59:59',
                        ]);
                    }

                    return $query;
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })
                ->addColumn('pickup_by', function ($row) {
                    return $row->pickupBy->display_name ?? 'N/A';
                })
                ->addColumn('delivery_by', function ($row) {
                    return $row->deliveryBy->display_name ?? 'N/A';
                })

                ->editColumn('completed_at', function ($row) {
                    if (! $row->completed_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->completed_at)->format('d-F-Y');
                })
                ->addColumn('status_label', function ($row) {
                    $status = DeliveryStatus::tryFrom($row->status);
                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                        DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                        DeliveryStatus::ASSIGNED => 'text-info',
                        DeliveryStatus::DELIVERED => 'text-success',
                        DeliveryStatus::COMPLETED => 'text-success',
                        DeliveryStatus::RETURNED => 'text-warning',
                        DeliveryStatus::CANCELED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                            <div class="badge space-x-2.5 {$colorClass}">
                                <div class="size-2 rounded-full bg-current"></div>
                                <span>{$label}</span>
                            </div>
                        HTML;
                })
                ->addColumn('status_clickable', function ($row) {
                    return DeliveryStatus::tryFrom($row->status) === DeliveryStatus::ASSIGNED;
                })

                ->rawColumns(['status_label', 'status_clickable'])
                ->make(true);

        }

        $deliveryBy = User::get()->pluck('display_name', 'id');
        $customer = Customer::active()
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();
        $statusOptions = collect(DeliveryStatus::cases())->map(function ($status) {
            return [
                'value' => $status->value,
                'label' => $status->label(),
            ];
        });

        return view('pages.reports.pickup-commission', compact('deliveryBy', 'customer', 'statusOptions'));
    }

    public function incomeShop(Request $request)
    {
        if ($request->ajax()) {
            $data = Delivery::where('branch_id', auth()->user()->default_branch_id)
                ->where('status', DeliveryStatus::COMPLETED->value)
                ->where('is_paid', true)
                ->when($request->delivery_by, function ($query, $delivery_by) {
                    return $query->where('delivery_by', $delivery_by);
                })
                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->when($request->status, function ($query, $status) {
                    return $query->where('status', $status);
                })
                ->when($request->completed_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->completed_at);

                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('completed_at', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        $date = $dates[0];

                        return $query->whereBetween('completed_at', [
                            $date.' 00:00:00',
                            $date.' 23:59:59',
                        ]);
                    }

                    return $query;
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })
                ->addColumn('pickup_by', function ($row) {
                    return $row->pickupBy->display_name ?? 'N/A';
                })
                ->addColumn('delivery_by', function ($row) {
                    return $row->deliveryBy->display_name ?? 'N/A';
                })

                ->editColumn('completed_at', function ($row) {
                    if (! $row->completed_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->completed_at)->format('d-F-Y');
                })
                ->addColumn('status_label', function ($row) {
                    $status = DeliveryStatus::tryFrom($row->status);
                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                        DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                        DeliveryStatus::ASSIGNED => 'text-info',
                        DeliveryStatus::DELIVERED => 'text-success',
                        DeliveryStatus::COMPLETED => 'text-success',
                        DeliveryStatus::RETURNED => 'text-warning',
                        DeliveryStatus::CANCELED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                            <div class="badge space-x-2.5 {$colorClass}">
                                <div class="size-2 rounded-full bg-current"></div>
                                <span>{$label}</span>
                            </div>
                        HTML;
                })
                ->addColumn('status_clickable', function ($row) {
                    return DeliveryStatus::tryFrom($row->status) === DeliveryStatus::ASSIGNED;
                })
                ->rawColumns(['status_label', 'status_clickable'])
                ->make(true);

        }

        $deliveryBy = User::get()->pluck('display_name', 'id');
        $customer = Customer::active()
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();
        $statusOptions = collect(DeliveryStatus::cases())->map(function ($status) {
            return [
                'value' => $status->value,
                'label' => $status->label(),
            ];
        });

        return view('pages.reports.income-shop', compact('deliveryBy', 'customer', 'statusOptions'));
    }
}
