<?php

namespace App\Http\Controllers;

use App\Http\Requests\ZonePrice\CreateRequest;
use App\Http\Requests\ZonePrice\UpdateRequest;
use App\Models\ZonePrice;
use App\Models\ZoneType;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class ZonePriceController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:zone-price.view')->only(['index', 'show']);
        $this->middleware('permission:zone-price.create')->only(['create', 'store']);
        $this->middleware('permission:zone-price.edit')->only(['edit', 'update']);
        $this->middleware('permission:zone-price.delete')->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = ZonePrice::with('branch', 'zoneType')
                ->where('branch_id', auth()->user()->default_branch_id)
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('branch', function ($row) {
                    return $row->branch->name ?? '-';
                })
                ->addColumn('zoneType', function ($row) {
                    return $row->zoneType->zone_type ?? '-';
                })
                ->addColumn('action', function ($row) {
                    return view('pages.configuration.zone-price.action-dropdown', compact('row'))->render();
                })
                ->make(true);
        }

        return view('pages.configuration.zone-price.index');

    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $zoneType = ZoneType::where('status', 'active')->get();

        return view('pages.configuration.zone-price.create', compact('zoneType'));

    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CreateRequest $request)
    {
        $validated = $request->validated();
        ZonePrice::create([
            'branch_id' => auth()->user()->default_branch_id,
            'zone_type_id' => $validated['zone_type'],
            'zone_name' => $validated['zone_name'],
            'price' => $validated['price'],
            'currency' => 'USD',
            'status' => $validated['status'],
        ]);

        return back()->with('success', 'Zone created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $zoneType = ZoneType::where('status', 'active')->get();
        $zone = ZonePrice::findOrFail($id);

        return view('pages.configuration.zone-price.edit', compact('zone', 'zoneType'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateRequest $request, ZonePrice $zonePrice)
    {
        $validated = $request->validated();

        $zonePrice->update([
            'zone_type_id' => $validated['zone_type'],
            'zone_name' => $validated['zone_name'],
            'price' => $validated['price'],
            'status' => $validated['status'],
        ]);

        return redirect()->route('zone-price.index')->with('success', 'Zone updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
